<?php

	namespace App\Services;

	use App\Event;
	use App\EventKind;
	use App\Location;

	class XMLEventExporter{
		/** @var \App\EventDate[] */
		protected $eventDates;
		
		protected $weekDays = [
			'SO',
			'MO',
			'DI',
			'MI',
			'DO',
			'FR',
			'SA',
		];

		protected $template = 'xmlExport.export_neu_liste';

		/**
		 * XMLEventExporter constructor.
		 * @param \App\EventDate[] $eventDates
		 * @param string|null $template
		 */
		public function __construct(array $eventDates, string $template = null){
			$this->setEventDates($eventDates);

			if($template){
				$this->setTemplate($template);
			}
		}

		/**
		 * @return \App\EventDate[]
		 */
		public function getEventDates(): array{
			return $this->eventDates;
		}

		/**
		 * @param \App\EventDate[] $eventDates
		 * @return XMLEventExporter
		 */
		public function setEventDates(array $eventDates): XMLEventExporter{
			$this->eventDates = $this->addData($eventDates);
			return $this;
		}

		/**
		 * @return string
		 */
		public function getTemplate(): string{
			return $this->template;
		}

		/**
		 * @param string $template
		 * @return XMLEventExporter
		 */
		public function setTemplate(string $template): XMLEventExporter{
			$this->template = $template;
			return $this;
		}
		
		protected function formatDate(string $date){
			return implode('.', array_reverse(explode('-', $date)));
		}

		protected function addData($eventDates){
			$deleted = [];
			$icons = $this->getIconArray();
			foreach($eventDates as $key => $eventDate){
				$event = Event::select()
					->where('id', '=', $eventDate->event)
					->get()
					->first();

				if($event === null){
					$deleted[] = $key;
					continue;
				}

				$eventKind = EventKind::select()
					->where('id', '=', $event->kind)
					->get()
					->first();
				$event->kind = $eventKind;

				$eventDate->event = $event;

				$location = Location::select()
					->where('id', '=', $eventDate->location)
					->get()
					->first();

				$eventDate->location = $location;
				
				if($eventDate->start){
					$time = explode(':',$eventDate->start);
					$eventDate->start = $time[0] . ':' . $time[1];
				}
				if($eventDate->end){
					$time = explode(':',$eventDate->end);
					$eventDate->end = $time[0] . ':' . $time[1];
				}
				
				$eventDate->dateParsed = $this->weekDays[date('w', strtotime($eventDate->date))]
					. ' ' . $this->formatDate($eventDate->date);
				if($eventDate->date_end && $eventDate->date_end !== $eventDate->date){
					$eventDate->dateParsed .= ' - ' . $this->weekDays[date('w', strtotime($eventDate->date_end))]
						. $this->formatDate($eventDate->date_end);
				}

				$eventDate->timeParsed = null;
				if($eventDate->start && !$eventDate->end){
					$eventDate->timeParsed = 'Ab ' . $eventDate->start;
				}

				if(!$eventDate->start && $eventDate->end){
					$eventDate->timeParsed = 'Bis ' . $eventDate->end;
				}

				if($eventDate->start && $eventDate->end){
					$eventDate->timeParsed = $eventDate->start . ' - ' . $eventDate->end;
				}

				$eventDate->event->kind->tag = $this->getCategoryTag($eventDate, $key);

				$eventDate->icon_list = '';
				
				if($eventDate->event->price){
					$eventDate->event->price = str_replace('</p>', "</p>\t", $eventDate->event->price);
					$eventDate->event->price = rtrim($eventDate->event->price);
				}
				
				$this->addFullLocationName($eventDate);
				
				if($eventDate->event->accessible){
					$eventDate->icon_list .= $icons['accessibility'];
				}
				if($eventDate->event->family_friendly){
					$eventDate->icon_list .= $icons['family'];
				}
				if($eventDate->event->sustainability){
					$eventDate->icon_list .= $icons['nature'];
				}
			}

			foreach($deleted as $key){
				unset($eventDates[$key]);
			}
			return $eventDates;
		}
		
		private function addFullLocationName($eventDate){
			$eventDate->fullLocation = $eventDate->location->name;
			if($eventDate->street){
				$eventDate->fullLocation.= ' - ' . $eventDate->street;
				if($eventDate->street){
					$eventDate->fullLocation.= ' ' . $eventDate->street_nr;
				}
			}
			
			if($eventDate->address_supplement){
					$eventDate->fullLocation.= ' ' . $eventDate->address_supplement;
			}
		}
		
		/**
		 * @return string
		 * @throws \Throwable
		 */
		public function generateXml(string $template = null): string{
			$template = $template ?? $this->template;
			$icons = $this->getIconArray();
			
			return view($template, ['eventDates' => $this->eventDates, 'icons' => $icons])->render();
		}

		protected function getCategoryTag($eventDate, $index): string{
			$kat = '';
			$tag = '_liste_';
			$index++;
			$head = '';
			switch($eventDate->event->kind->id){
				case 1://kreuz und quer
					$tag .= 'kreuz';
					$kat .= '10';
					$head.= '10_Kreuz';
					break;
				case 3://Musik und Tanz
					$tag .= 'musik';
					$kat .= '1';
					$head.= '1_Musik';
					break;
				case 502://Märkte und messen
					$tag .= 'maerkte';
					$kat .= '2';
					$head.= '2_Maerkte';
					break;
				case 503://Kunst und Ausstellungen
					$tag .= 'kunst';
					$kat .= '3';
					$head.= '3_Kunst';
					break;
				case 504://Freizeit und Sport
					$tag .= 'freizeit';
					$kat .= '4';
					$head.= '4_Freizeit';
					break;
				case 505://Führungen und Wanderungen
					$tag .= 'fuehrungen';
					$kat .= '5';
					$head.= '5_Fuehrungen';
					break;
				case 506://Kulinarik und Genuss
					$tag .= 'kulinarik';
					$kat .= '6';
					$head.= '6_Kulinarik';
					break;
				case 507://Kirchweih und Feste
					$tag .= 'kirchweih';
					$kat .= '7';
					$head.= '7_Kirchweih';
					break;
				case 508://Theater Kabarett Kino
					$tag .= 'theater';
					$kat .= '8';
					$head.= '8_Theater';
					break;
				case 509://Seminare Vorträge beratung
					$tag .= 'seminare';
					$kat .= '9';
					$head.= '9_Seminare';
					break;
			}

			$eventDate->index = $kat;

			return $head;
		}

		protected function getIconArray(){
			return array(
				'location' => '',
				'date' => '',
				'time' => '',
				'price' => '',
				'accessibility' => '',
				'family' => '',
				'nature' => '',
				'organisator' => '',
				'url' => '',
				'mail' => '',
				'phone' => '',
			);
		}
	}
