<?php

	namespace App\Http\Controllers;

	use App\ApiUser;
	use App\Http\Resources\LoggedInUser;
	use App\Http\Resources\LraUser;
	use App\Services\ApiKeyGenerator;
	use App\Services\Validators\UserChecker;
	use App\User;
	use App\UserGroup;
	use Illuminate\Http\Request;
	use Illuminate\Support\Facades\DB;
	use Illuminate\Support\Facades\Hash;

	class UserController extends Controller{

		const ROUTE_GROUP_LRA = 1;
		const ROUTE_GROUP_USER = 2;

		/**
		 * @var \App\Services\ApiKeyGenerator
		 */
		protected $apiKeyGenerator;

		public function __construct(ApiKeyGenerator $apiKeyGenerator){
			$this->apiKeyGenerator = $apiKeyGenerator;
		}

		/**
		 * Display a listing of the resource.
		 *
		 * @param Request $request
		 * @return \Illuminate\Http\JsonResponse
		 */
		public function login(Request $request){
			$params = $request->all();
			if(empty($params['user']) || empty($params['password'])){
				return response()->json(null, 400);
			}

			$user = User::select()->where([
				'email' => $params['user'],
			])->first();

			if(!$user){
				$user = User::select()->where([
					'name' => $params['user'],
				])->first();
			}
			if($user && password_verify($params['password'], $user->password)){
				$api_user = ApiUser::find($user->api_user);

				if(
					!$api_user
					|| $api_user->domain !== parse_url(((isset($_SERVER['HTTP_ORIGIN']) && $_SERVER['HTTP_ORIGIN'])? $_SERVER['HTTP_ORIGIN'] : 'http://localhost'), \PHP_URL_HOST)
				){
					return response()->json(false, 400);
				}


				return response()->json(new LoggedInUser($user), 200);
			}

			return response()->json(false, 400);
		}

		public function index(Request $request){
			$forApiUsers = DB::table('api_users')
				->select('id')
				->where('group_id', '=', $request->currentApiUser->group_id)
				->get()
				->all();
			$apiUsers = [];

			foreach($forApiUsers as $apiUser){
				$apiUsers[] = $apiUser->id;
			}

			$query = User::query()
				->whereIn('api_user', $apiUsers);

			foreach($request->filter as $value){
				if(in_array($value[0], User::$filterable)){
					$query->where(...$value);
				}
			}

			$query = $query->paginate($request->pageSize)
				->appends(request()->query());

			return \App\Http\Resources\User::collection($query);
		}

		/**
		 * Display the specified resource.
		 *
		 * @param User $user
		 * @return \Illuminate\Http\JsonResponse
		 */
		public function show(User $user){
			return response()->json(new \App\Http\Resources\User($user), 200);
		}

		/**
		 * Store a newly created resource in storage.
		 *
		 * @param \Illuminate\Http\Request $request
		 * @return \Illuminate\Http\JsonResponse
		 */
		public function store(Request $request, UserChecker $userValidator){
			$params = $request->json()->all();

			$errors = $userValidator->validate($params);

			if(count($errors) === 0){
				$errors = $userValidator->checkDoubleUser($params);
			}

			if(empty($params['password'])){
				$errors[] = 'Passwort fehlt';
			}

			if(count($errors)>0){
				return response()->json(['errors' => $errors], 200);
			}

			$params['password'] = Hash::make($params['password']);

			$newUser = User::create($params);

			$newApiUser = ApiUser::create([
				'group_id' => $request->currentApiUser->group_id,
				'route_group' => 2,
				'key' => $this->apiKeyGenerator->generate(),
				'domain' => $params['domain'],
			]);

			$newUser->api_user = $newApiUser->id;
			$newUser->save();


			return response()->json(new \App\Http\Resources\User($newUser), 201);
		}

		/**
		 * Update the specified resource in storage.
		 *
		 * @param \Illuminate\Http\Request $request
		 * @param User $user
		 * @return \Illuminate\Http\JsonResponse
		 */
		public function update(Request $request, User $user, UserChecker $userValidator){
			$params = $request->all();

			$errors = $userValidator->validate($params);
			if(count($errors)>0){
				return response()->json(['errors' => $errors], 200);
			}

			$user->name = $params['name'];
			$user->email = $params['email'];

			if($params['password']){
				$user->password = Hash::make($params['password']);
			}

			$user->save();
			return response()->json(new \App\Http\Resources\User($user), 200);
		}

		/**
		 * Remove the specified resource from storage.
		 *
		 * @param User $user
		 * @return \Illuminate\Http\JsonResponse
		 */
		public function destroy(Request $request, User $user){
			if($user->api_user !== $request->currentApiUser->id){
				$user->delete();
			}else{
				return response()->json(['error' => 'Der eigene User kann nicht gelöscht werden']);
			}

			return response()->json(null, 204);
		}

		public function lraIndex(Request $request){
			$query = User::select();

			foreach($request->filter as $value){
				if(in_array($value[0], User::$filterable)){
					$query->orWhere(...$value);
				}
			}

			$query->orderBy('name', 'asc');


			$query = $query->paginate($request->pageSize)
				->appends(request()->query());

			return LraUser::collection($query);
		}

		/**
		 * Display the specified resource.
		 *
		 * @param User $user
		 * @return \Illuminate\Http\JsonResponse
		 */
		public function lraShow(User $user){
			return response()->json(new LraUser($user), 200);
		}

		public function lraStore(Request $request, UserChecker $userValidator){
			$params = $request->json()->all();

			$errors = $userValidator->validate($params);

			if(count($errors) === 0){
				$errors = $userValidator->checkDoubleUser($params);
			}

			if(empty($params['password'])){
				$errors[] = 'Passwort fehlt';
			}

			if(count($errors)>0){
				return response()->json(['errors' => $errors], 200);
			}

			$params['password'] = Hash::make($params['password']);

			$newUser = User::create($params);

			$newApiUser = ApiUser::create([
				'group_id' => $params['group_id'],
				'route_group' => $params['group_id'] === 1? self::ROUTE_GROUP_LRA : self::ROUTE_GROUP_USER,
				'key' => $this->apiKeyGenerator->generate(),
				'domain' => $params['domain'],
			]);

			$newUser->api_user = $newApiUser->id;
			$newUser->save();


			return response()->json(new LraUser($newUser), 201);
		}

		/**
		 * Update the specified resource in storage.
		 *
		 * @param \Illuminate\Http\Request $request
		 * @param User $user
		 * @return \Illuminate\Http\JsonResponse
		 */
		public function lraUpdate(Request $request, User $user, UserChecker $userValidator){
			$params = $request->all();

			$errors = $userValidator->validate($params);

			if(count($errors)>0){
				return response()->json(['errors' => $errors], 200);
			}

			$user->name = $params['name'];
			$user->email = $params['email'];

			if($params['password']){
				$user->password = Hash::make($params['password']);
			}

			$user->save();

			$apiUser = ApiUser::find($user->api_user);
			if($apiUser->group_id != $params['group_id']){
				$apiUser->route_group = $params['group_id'] === 1? self::ROUTE_GROUP_LRA : self::ROUTE_GROUP_USER;
			}
			$apiUser->group_id = $params['group_id'];
			$apiUser->domain = $params['domain'];

			$apiUser->save();

			return response()->json(new LraUser($user), 200);
		}

		/**
		 * Remove the specified resource from storage.
		 *
		 * @param User $user
		 * @return \Illuminate\Http\JsonResponse
		 */
		public function lraDestroy(Request $request, User $user){
			if($user->api_user !== $request->currentApiUser->id){
				$user->delete();
			}else{
				return response()->json(['error' => 'Der eigene User kann nicht gelöscht werden']);
			}

			return response()->json(null, 204);
		}

	}
