<?php

	namespace App\Http\Controllers;

	use App\Http\Resources\Location as LocationResource;
	use App\Location;
	use App\Services\Validators\LocationChecker;
	use Grimzy\LaravelMysqlSpatial\Types\Point;
	use Illuminate\Http\Request;

	class LocationController extends Controller{
		/**
		 * @param Request $request
		 * @return \Illuminate\Http\Resources\Json\AnonymousResourceCollection
		 */
		public function index(Request $request){
			$query = Location::select()->with(['children', 'parent']);
			foreach($request->filter as $value){
				if(in_array($value[0], Location::$filterable)){
					$query->where(...$value);
				}
			}

			$query->orderBy('name', 'asc');

			if($request->pageSize){
				$query = $query->paginate($request->pageSize);
			}
			$query = $query->appends(request()->query());

			if($request->withoutChildren){
				return LocationResource::collection($query);
			}
			$coll = LocationResource::collection($query);
			$ids = [];
			foreach($coll as $loc){
				$ids[] = $loc->id;
				if($loc->resource->children){
					foreach($loc->resource->children as $child){
						$ids[] = $child->id;
					}
				}
			}

			$ids = array_unique($ids);

			$query = Location::select()->with(['children']);
			$query->whereIn('id', $ids);
			$query->orderBy('locations.name', 'asc');
			if($request->pageSize){
				$query = $query->paginate($request->pageSize);
			}
			$query = $query->appends(request()->query());

			$coll = LocationResource::collection($query);
			return $coll;
		}

		public function show(Location $location){
			return response()->json(new LocationResource($location), 200);
		}


		public function store(Request $request, LocationChecker $locationValidator){
			$params = $request->all();

			$errors = $locationValidator->validate($params);
			if(count($errors)>0){
				return response()->json(['errors' => $errors],200);
			}

			if(empty($params['gps']['lat']) || empty($params['gps']['lng'])){
				unset($params['gps']);
			}else{
				$params['gps'] = new Point($params['gps']['lat'], $params['gps']['lng']);
			}

			if($params['parent']){
				$params['parent_id'] = $params['parent'];
				unset($params['parent']);
			}


			$location = Location::create($params);

			return response()->json($location, 201);
		}

		public function update(Request $request, Location $location, LocationChecker $locationValidator){
			$params = $request->all();

			$errors = $locationValidator->validate($params);
			if(count($errors)>0){
				return response()->json(['errors' => $errors],200);
			}

			if(empty($params['gps']['lat']) || empty($params['gps']['lng'])){
				unset($params['gps']);
			}

			$parentPos = strpos($params['name'], '(');

			if($parentPos !== false){
				$params['name'] = substr($params['name'], 0, $parentPos - 1);
			}

			$forUpdate = [
				'name' => $params['name'],
				//'gps' => new Point($params['gps']['lat'], $params['gps']['lng']),
			];
			if(isset($params['gps'])){
				$forUpdate['gps'] = new Point($params['gps']['lat'], $params['gps']['lng']);
			}

			if(isset($params['parent'])){
				//check for infinit recusion

				$parentId = $params['parent'];
				$ids = [];
				do{
					$parent = Location::where('id', '=', $parentId)->get()->first();
					if(!$parent){
						break;
					}
					$ids[] = $parentId;
					$parentId = $parent->parent_id;
				}while($parentId);

				$ids[] = $location->id;

				if(count(array_unique($ids)) !== count($ids)){
					return response()->json(['errors' => ['Endlosverschachtelung']], 200);
				}

				$forUpdate['parent_id'] = $params['parent']? $params['parent'] : null;
			}

			$location->update($forUpdate);

			return response()->json(new LocationResource($location), 200);
		}


		public function destroy(Location $location){
			$location->delete();

			return response()->json(null, 204);
		}
	}
