<?php

	namespace App\Http\Controllers;

	use App\EventDate;
	use App\Http\Resources\FullEvent;
	use App\Location;
	use Illuminate\Http\Request;
	use Illuminate\Support\Facades\DB;

	class EventDateController extends Controller{

		protected $eventDateService;

		public function __construct(\App\Services\EventDate $eventDateService){
			$this->eventDateService = $eventDateService;
		}

		/**
		 * @param \Illuminate\Http\Request $request
		 */
		public function index(Request $request){
			$query = EventDate::select('event_dates.id');

			$query->join('events', 'event_dates.event', '=', 'events.id')
				->join('event_releases', function($join) use ($request){
					$join->on('event_releases.event', '=', 'event_dates.event')
						->where('event_releases.user_group', '=', $request->currentApiUser->group_id);
				})
				->join('locations', 'event_dates.location', '=', 'locations.id');

			$query->where('event_releases.released', '=', 1);

			$allowed_filters = [
				'event_dates.date',
				'events.kind',
				'event_dates.location',
				'event_dates.title',
				'searchTerm',
			];

			$eventDateQueries = [];

			foreach($request->filter as $value){
				if(in_array($value[0], $allowed_filters)){
					if($value[0] == 'event_dates.location'){
						$value[2] = array_merge($value[2], Location::getAllChildren($value[2]));
					}

					if(strtolower($value[1]) === 'in'){
						$query->whereIn($value[0], $value[2]);
					}else{
						if(strtolower($value[0]) === 'searchterm'){
							$query->where(function($query) use ($value){
								$query->orWhere('events.name', 'like', $value[2]);
								$query->orWhere('event_dates.title', 'like', $value[2]);
								$query->orWhere('event_dates.description', 'like', $value[2]);
							});
						}else{
							if($value[0] !== 'event_dates.date'){
								$query->where(...$value);
							}else{
								$eventDateQueries[] = $value;
							}
						}
					}
				}
			}

			if(count($eventDateQueries) > 0){
				$this->eventDateService->mergeEventDateQueries($query, $eventDateQueries);
			}

			$query->orderBy('date', 'asc');
			$query->orderBy('start', 'asc');

			$query->groupBy('event_dates.id');

			$query = $query->paginate($request->pageSize)
				->appends(request()->query());

			return FullEvent::collection($query);
		}

		public function show(Request $request, EventDate $eventDate){
			$release = DB::table('event_releases')
				->where([
					['event', '=', $eventDate->event],
					['user_group', '=', $request->currentApiUser->group_id],
				])->get()->first();

			if(!$release){
				return response()->json(false, 404);
			}
			return response()->json(new FullEvent($eventDate), 200);
		}

		/**
		 * @return \Illuminate\Http\JsonResponse
		 */
		public function getYears(){
			$years = DB::table('event_dates')
				->select(DB::raw('YEAR(date) as year'))
				->distinct('year')
				->get()
				->map(function($val){
					return $val->year;
				});

			return response()->json($years, 200);
		}

		public function downloadAsXML(Request $request){
			$events = \App\EventDate::select('event_dates.*');

			$group = $request->get('group', false);

			if($group && $group !== 'null'){
				$events->join('event_releases', function($join) use ($request){
					$join->on('event_releases.event', '=', 'event_dates.event')
						->where('event_releases.user_group', '=', $request->get('group'));
				})
					->join('events', 'event_dates.event', '=', 'events.id')
					->where('event_releases.released', '=', 1);
			}

			$allowed_filters = [
				'event_dates.date',
				'events.kind',
				'event_dates.location',
				'event_dates.title',
				'searchTerm',
			];

			foreach($request->filter as $value){
				if(in_array($value[0], $allowed_filters)){
					$events->where(...$value);
				}
			}
			$template = null;
			switch($request->get('sortOrder')){
				case 'date':
					$events = $events->orderBy('event_dates.date', 'asc')
						->orderBy('event_dates.start', 'asc');
					$template = 'xmlExport.export_neu_liste';
					break;

				case 'category':
					$events = $events->orderBy('events.kind', 'asc')
						->orderBy('event_dates.date', 'asc')
						->orderBy('event_dates.start', 'asc');
					$template = 'xmlExport.export_neu_category';
					break;
			}

			//$template = 'xmlExport.export_second';
			$template = 'xmlExport.export_neu_2023';

			$events = $events->get()
				->all();

			$exporter = new \App\Services\XMLEventExporter($events, $template);

			$xml = $exporter->generateXml();

			return response()
				->streamDownload(function() use ($xml){
					//$xml = preg_replace('/\s+<event_icon>/', '<event_icon>', $xml);
					$xml = str_replace(['&nbsp;', ' <event_icon>',  ' $',' ___remove___ '], [' ', '<event_icon>', '$', ''], $xml);
					echo str_replace(['&nbsp;', ' <event_icon>',  ' $',], [' ', '<event_icon>', '$',], $xml);
				}, 'veranstaltungsexport.xml');
		}

		public function downloadAsCsv(Request $request){
			$events = \App\EventDate::select('event_dates.*');

			$group = $request->get('group', false);

			if($group && $group !== 'null'){
				$events->join('event_releases', function($join) use ($group){
					$join->on('event_releases.event', '=', 'event_dates.event')
						->where('event_releases.user_group', '=', $group);
				})
					->where('event_releases.released', '=', 1);
			}else{
				return response()->json(['error' => 'missing_group'], 400);
			}

			$events->join('events', 'event_dates.event', '=', 'events.id');

			$allowed_filters = [
				'event_dates.date',
				'events.kind',
				'event_dates.location',
				'event_dates.title',
				'searchTerm',
			];
			foreach($request->filter as $value){
				if(in_array($value[0], $allowed_filters, true)){
					if($value[0] === 'event_dates.location'){
						$value[2] = array_merge([$value[2]], Location::getAllChildren([$value[2]]));
					}

					if(strtolower($value[1]) === 'in'){
						$events->whereIn($value[0], $value[2]);
					}else{
						if(strtolower($value[0]) === 'searchterm'){
							$events->where(function($query) use ($value){
								$query->orWhere('events.name', 'like', $value[2]);
								$query->orWhere('event_dates.title', 'like', $value[2]);
								$query->orWhere('event_dates.description', 'like', $value[2]);
							});
						}else{
							$events->where(...$value);
						}
					}
				}
			}

			switch($request->get('sortOrder')){
				case 'date':
					$events = $events->orderBy('event_dates.date', 'asc')
						->orderBy('event_dates.start', 'asc');;
					break;

				case 'category':
					$events = $events->orderBy('events.kind', 'asc')
						->orderBy('event_dates.date', 'asc')
						->orderBy('event_dates.start', 'asc');
					break;
			}

			$events = $events->get()
				->all();
			$header = [
				'Veranstaltungs-ID',
				'Veranstaltungstitel',
				'Titelzusatz',
				'Beschreibung',
				'Veranstaltungsart',
				'Veranstalter',
				'Web',
				'Email',
				'Telefon',
				'Anmeldung',
				'Preis',
				'Ticketshop',
				'Kostenpflichtig',
				'Facebook',
				'Twitter',
				'Instagram',
				'Nachhaltig',
				'Barrierefrei',
				'Familienfreundlich',
				'ID Veranstaltungsdatum',
				'Ort',
				'GPS',
				'Straße',
				'Hausnummer',
				'Adresszusatz',
				'Abgesagt',
				'Veranstaltungsdatum',
				'Datum Veranstaltungsende',
				'Uhrzeit Beginn',
				'Uhrzeit Ende',
				'Ganztags',
			];
			$exporter = new \App\Services\CSVEventExporter($header, $events);

			$csv = $exporter->generateCsv();

			return response()->streamDownload(function() use ($csv){
				echo $csv;
			}, 'export.csv', [
				'Content-Type' => 'text/csv; charset=utf-8',
				'Content-Disposition' => 'attachment; filename="export.csv"',
			]);
		}

		public function locationGPS(Request $request, \App\Services\GPSApiService $gps_service){
			if($request->get('searchString', '')){
				$gps = $gps_service->getLocation($request->get('searchString'));
			}else{
				$gps = [
					'lat' => '',
					'lng' => '',
				];
			}

			return response()->json($gps, 200);
		}
	}
