<?php

	namespace App\Services;

	use App\Event;
	use App\EventKind;
	use App\Location;

	class CSVEventExporter{
		/** @var String[] */
		protected $header;

		/** @var \App\EventDate[] */
		protected $eventDates;

		/**
		 * XMLEventExporter constructor.
		 * @param String[] $header
		 * @param \App\EventDate[] $eventDates
		 */
		public function __construct(array $header, array $eventDates){
			$this->header = $header;
			$this->setEventDates($eventDates);
		}

		/**
		 * @return \App\EventDate[]
		 */
		public function getEventDates(){
			return $this->eventDates;
		}

		/**
		 * @param \App\EventDate[] $eventDates
		 * @return $this
		 */
		public function setEventDates(array $eventDates){
			$this->eventDates = $this->addData($eventDates);
			return $this;
		}

		/**
		 * @return String[]
		 */
		public function getHeader(){
			return $this->header;
		}

		/**
		 * @param String[] $header
		 * @return $this
		 */
		public function setHeader(array $header){
			$this->header = $header;
			return $this;
		}

		protected function addData($eventDates){
			$deleted = [];

			foreach($eventDates as $key => $eventDate){
				$event = Event::select()
					->where('id', '=', $eventDate->event)
					->get()
					->first();

				if($event === null){
					$deleted[] = $key;
					continue;
				}

				$eventKind = EventKind::select()
					->where('id', '=', $event->kind)
					->get()
					->first();
				$event->kind = $eventKind;

				$eventDate->event = $event;

				$location = Location::select()
					->where('id', '=', $eventDate->location)
					->get()
					->first();

				$eventDate->location = $location;


			}

			foreach($deleted as $key){
				unset($eventDates[$key]);
			}
			return $eventDates;
		}

		/**
		 * @return String
		 */
		public function generateCsv(): string{
			$result = '';

			foreach($this->header as $header){
				$result .= $this->escape($header) . ';';
			}

			$result = substr($result, 0, -1) . "\n";

			foreach($this->eventDates as $eventDate){
				$result .= $this->escape($eventDate->event->id) . ';';
				$result .= $this->escape($eventDate->event->name) . ';';
				$result .= $this->escape($eventDate->title) . ';';
				$result .= $this->escape(strip_tags($eventDate->description)) . ';';
				$result .= $this->escape($eventDate->event->kind->name) . ';';
				$result .= $this->escape($eventDate->event->organisation) . ';';
				$result .= $this->escape($eventDate->event->web) . ';';

				$result .= $this->escape($eventDate->event->mail) . ';';
				$result .= $this->escape($eventDate->event->phone) . ';';
				$result .= $this->escape($eventDate->event->registration? 'Ja' : 'Nein') . ';';
				$result .= $this->escape(html_entity_decode(strip_tags($eventDate->event->price))) . ';';
				$result .= $this->escape($eventDate->event->ticket_presale) . ';';
				$result .= $this->escape($eventDate->event->with_cost? 'Ja' : 'Nein') . ';';

				$result .= $this->escape($eventDate->event->facebook) . ';';
				$result .= $this->escape($eventDate->event->twitter) . ';';
				$result .= $this->escape($eventDate->event->instagram) . ';';

				$result .= $this->escape($eventDate->event->sustainability? 'Ja' : 'Nein') . ';';
				$result .= $this->escape($eventDate->event->accessible? 'Ja' : 'Nein') . ';';
				$result .= $this->escape($eventDate->event->family_friendly? 'Ja' : 'Nein') . ';';

				$result .= $this->escape($eventDate->id) . ';';
				$result .= $this->escape($eventDate->location->name) . ';';
				/** @var \Grimzy\LaravelMysqlSpatial\Types\Point? $gps */
				$gps = $eventDate->location->gps;

				if($gps && $gps->getLat() && $gps->getLng()){
					$result .= $this->escape('Lat: ' . $gps->getLat() . ' ' . 'Lng: ' . $gps->getLng()) . ';';
				}else{
					$result .= $this->escape(null) . ';';
				}

				$result .= $this->escape($eventDate->street) . ';';
				$result .= $this->escape($eventDate->street_nr) . ';';
				$result .= $this->escape($eventDate->address_supplement) . ';';

				$result .= $this->escape($eventDate->canceled? 'Ja' : 'Nein') . ';';

				$result .= $this->escape(implode('.', array_reverse(explode('-', $eventDate->date)))) . ';';
				$result .= $this->escape(implode('.', array_reverse(explode('-', $eventDate->date_end)))) . ';';

				$start_time = $eventDate->start;
				if($start_time){
					$start_time = substr($start_time, 0, strrpos($start_time, ':'));
				}

				$result .= $this->escape($start_time) . ';';

				$end_time = $eventDate->end;
				if($end_time){
					$end_time = substr($end_time, 0, strrpos($end_time, ':'));
				}


				$result .= $this->escape($end_time) . ';';
				$result .= $this->escape($eventDate->whole_day? 'Ja' : 'Nein') . ';';


				$result .= "\n";
			}


			return $result;
		}

		/**
		 * @param string|null $part
		 * @return string
		 */
		protected function escape(?string $part): string{
			return '"' . str_replace('"', '""', $part? $part : '') . '"';
		}

	}
