<?php

	namespace App\Http\Controllers;

	use App\Event;
	use App\EventDate;
	use App\Http\Resources\Event as EventResource;
	use App\Http\Resources\Usergroup;
	use App\Location;
	use App\Media;
	use App\Recommendation;
	use App\Services\ReleaseService;
	use App\Services\Validators\EventChecker;
	use App\UserGroupAuthority;
	use Grimzy\LaravelMysqlSpatial\Types\Point;
	use Illuminate\Http\JsonResponse;
	use Illuminate\Http\Request;
	use Illuminate\Http\Resources\Json\AnonymousResourceCollection;
	use Illuminate\Support\Facades\DB;

	class EventController extends Controller{

		const EVENT_FIELDS = '
			events.id,
			events.name,
			events.kind,
			events.status,
			events.organisation,
			events.web,
			events.mail,
			events.phone,
			events.registration,
			events.price,
			events.free,
			events.created_by
		 ';

		/**
		 * @var ReleaseService
		 */
		protected $releaseService;

		public function __construct(ReleaseService $releaseService){
			$this->releaseService = $releaseService;
		}

		/**
		 * Display a listing of the resource.
		 *
		 * @param Request $request
		 * @return AnonymousResourceCollection
		 */
		public function index(Request $request){
			$user_group = $request->currentApiUser->group_id;

			$query = Event::selectRaw(self::EVENT_FIELDS . ',event_dates.date');

			$query->leftJoin('recommendations', 'events.id', '=', 'recommendations.event');
			$query->leftJoin('event_dates', 'events.id', '=', 'event_dates.event');
			$query->leftJoin('event_releases', 'events.id', '=', 'event_releases.event');

			$allowedFilters = [
				'events.kind',
				'events.name',
				'event_dates.date',
				'event_dates.date_end',
				'event_dates.location',
				'event_releases.released',
				'created_by',
			];

			$allowedFilters = array_merge(Event::$filterable, $allowedFilters);
			foreach($request->filter as $value){
				if(in_array($value[0], $allowedFilters)){
					if($value[0] === 'event_dates.location' && $request->get('includeSubLocations', false)){
						$locations = Location::select('id')
							->where('parent_id', '=', $value[2])
							->get()
							->all();
						$locations = array_map(function($loc){
							return $loc->id;
						}, $locations);
						$locations = array_merge($locations, [$value[2]]);

						$query->whereIn($value[0], $locations);

						continue;
					}


					if($value[0] == 'event_releases.released'){

						if($value[2] == '1'){
							$query->where('event_releases.user_group', '=', $user_group);
						}else{
							$query->where(function($query){
								$query->orWhere('event_releases.user_group', 'IS', 'NULL');
								$query->orWhere('event_releases.user_group', '=', 0);
							});


						}
					}

					if(strtolower($value[1]) === 'is' && $value[2] === null){
						$query->whereNull($value[0]);
						continue;
					}

					$query->where(...$value);
				}
			}

			$query->whereNull('event_dates.deleted_at');

			$query->groupByRaw(self::EVENT_FIELDS . ',event_dates.date');

			$query->orderBy('event_dates.date', 'desc');
			$query = $query->paginate($request->pageSize)
				->appends(request()->query());

			return EventResource::collection($query);
		}

		/**
		 * @param Event $event
		 * @return JsonResponse
		 */
		public function show(Event $event){
			return response()->json(new EventResource($event), 200);
		}

		/**
		 * Store a newly created resource in storage.
		 *
		 * @param Request $request
		 * @return JsonResponse
		 */
		public function store(Request $request, EventChecker $eventChecker){
			$params = $request->json()->all();
			$eventErrors = $eventChecker->validateEvent($params);
			$eventDateErrors = [];
			if(isset($params['eventDates']) && is_array($params['eventDates']) && count($params['eventDates']) > 0){

				foreach($params['eventDates'] as $eventDate){
					$eventDateErrors = array_merge($eventDateErrors, $eventChecker->validateEventDate($eventDate));
				}
			}

			$errors = array_merge($eventErrors, $eventDateErrors);

			if(count($errors)){
				return response()->json(['errors' => $errors], 200);
			}

			if(isset($params['kind']) && is_array($params['kind']) && isset($params['kind']['id'])){
				$params['kind'] = $params['kind']['id'];
			}

			$params['created_by'] = $request->currentApiUser->group_id;

			$event = Event::create($params);

			if(isset($params['eventDates']) && is_array($params['eventDates']) && count($params['eventDates']) > 0){
				foreach($params['eventDates'] as $eventDate){
					if(empty($eventDate['canceled'])){
						$eventDate['canceled'] = 0;
					}
					$eventDate['event'] = $event->id;
					if(is_array($eventDate['location']) && isset($eventDate['location']['id'])){
						$eventDate['location'] = $eventDate['location']['id'];
					}else if(empty($eventDate['location'])){
						$eventDate['location'] = null;
					}

					if(empty($eventDate['gps']['lat']) || empty($eventDate['gps']['lng'])){
						unset($eventDate['gps']);
					}else{
						$eventDate['gps'] = new Point($eventDate['gps']['lat'], $eventDate['gps']['lng']);
					}

					EventDate::create($eventDate);
				}
			}
			$this->releaseService->updateReleases($event, $params['releases']);
			return response()->json(new EventResource($event), 201);
		}


		/**
		 * Update the specified resource in storage.
		 *
		 * @param Request $request
		 * @param Event $event
		 * @return JsonResponse
		 */
		public function update(Request $request, Event $event, EventChecker $eventChecker){
			$params = $request->all();
			$eventErrors = $eventChecker->validateEvent($params);
			$eventDateErrors = [];
			if(isset($params['eventDates']) && is_array($params['eventDates']) && count($params['eventDates']) > 0){

				foreach($params['eventDates'] as $eventDate){
					$eventDateErrors = array_merge($eventDateErrors, $eventChecker->validateEventDate($eventDate));
				}
			}

			$errors = array_merge($eventErrors, $eventDateErrors);

			if(count($errors)){
				return response()->json(['errors' => $errors], 200);
			}
			$event->update($params);

			if(isset($params['eventDates']) && is_array($params['eventDates']) && count($params['eventDates']) > 0){
				$event->updateEventDates($params['eventDates']);
			}else{
				$event->removeEventDate();
			}
			$this->releaseService->updateReleases($event, $params['releases']);


			return response()->json(new EventResource($event), 200);
		}

		/**
		 * Remove the specified resource from storage.
		 *
		 * @param Event $event
		 * @return JsonResponse
		 */
		public function destroy(Event $event){
			$this->releaseService->remove($event);

			EventDate::where('event', '=', $event->id)
				->delete();

			$event->delete();

			return response()->json(null, 204);
		}

		public function releases(Request $request){
			$userGroupAuth = UserGroupAuthority::query()
				->where('user_group_main', '=', $request->currentApiUser->group_id)
				->get()
				->all();
			$userGroupAuth = array_map(function($auth){
				return \App\UserGroup::find($auth->user_group_child);
			}, $userGroupAuth);

			$params = $request->all();
			$releases = [];
			if(isset($params['event']) && $params['event']){
				foreach(array_merge([$request->currentApiUser->group_id], $userGroupAuth) as $auth){
					$release = DB::table('event_releases')
						->select('released')
						->where([
							['event', '=', $params['event']],
							['user_group', '=', is_object($auth)? $auth->id : $auth],
						])
						->get()
						->first();
					$release = $release? $release->released : 0;
					$releases[is_object($auth)? $auth->id : $auth] = $release;
				}
			}

			return response()->json([
				'mainGroup' => new Usergroup(\App\UserGroup::find($request->currentApiUser->group_id)),
				'authGroups' => Usergroup::collection($userGroupAuth),
				'releases' => $releases,
			], 200);
		}

		public function updateMedia(Event $event, Request $request){
			$params = $request->all();

			$event->detachAllMedia();
			$apiMedia = ((isset($params['apiMedia']))? $params['apiMedia'] : []);
			foreach($apiMedia as $media){
				$event->attachMedia($media);
			}

			if(isset($request->files) && $request->files->count() > 0){
				$files = $request->files->get('uploadedMedia');
				foreach($files as $file){
					$newFile = new \Illuminate\Http\UploadedFile(
						$file->getRealPath(),
						$file->getClientOriginalName(),
						$file->getClientMimeType()
					);

					$newMedia = Media::createFromFile($newFile, Event::getStorageFolder(), $request);

					$event->attachMedia($newMedia);
				}
			}

			return response()->json(null, 204);
		}

		public function myEvents(Request $request){
			$user_group = $request->currentApiUser->group_id;

			$query = Event::selectRaw(self::EVENT_FIELDS . ',event_dates.date');

			$hasReleasedFilter = false;

			$allowedFilters = [
				'events.name',
				'events.kind',
				'event_dates.date',
				'event_dates.date_end',
				'event_dates.location',
				'event_releases.released',
				'events.created_by',
			];

			$allowedFilters = array_merge(Event::$filterable, $allowedFilters);
			foreach($request->filter as $value){
				if(in_array($value[0], $allowedFilters)){
					if($value[0] === 'event_dates.location' && $request->get('includeSubLocations', false)){
						$locations = Location::select('id')
							->where('parent_id', '=', $value[2])
							->get()
							->all();
						$locations = array_map(function($loc){
							return $loc->id;
						}, $locations);
						$locations = array_merge($locations, [$value[2]]);

						$query->whereIn($value[0], $locations);

						continue;
					}

					if($value[0] === 'event_releases.released'){
						$hasReleasedFilter = true;
						$query->where(function($query) use($value, $user_group){

							if($value[2] == '0'){
								$query->where(function($query) use ($user_group){
									$query->orWhere('event_releases.user_group', '=', $user_group);
									$query->orWhereNull('event_releases.user_group');
								});
								$query->where(function($query){
									$query->orWhereNull('event_releases.released');
									$query->orWhere('event_releases.released', '=', 0);
								});
							}else{
								$query->where('event_releases.user_group', '=', $user_group);
							}
						});
						continue;
					}

					if(strtolower($value[1]) === 'is' && $value[2] === null){
						$query->whereNull($value[0]);
						continue;
					}

					$query->where(...$value);
				}
			}

			$query->where(function($query) use ($user_group, $hasReleasedFilter){
				$query->orWhere('events.created_by', '=', $user_group);
				if(!$hasReleasedFilter){
					$query->orWhere(function($query) use ($user_group){
						$query->where([
							['event_releases.user_group', '=', $user_group],
							['event_releases.released', '=', '1'],
						]);

					});
				}
			});

			$query->whereNull('event_dates.deleted_at');

			$query->leftJoin('event_dates', 'events.id', '=', 'event_dates.event');
			$query->leftJoin('event_releases', 'events.id', '=', 'event_releases.event');

			$query->groupByRaw(self::EVENT_FIELDS . ',event_dates.date');

			$query->orderBy('event_dates.date', 'desc');

			//print_r(\Illuminate\Support\Str::replaceArray('?', $query->getBindings(), $query->toSql()));
			//die();
			$query = $query->paginate($request->pageSize)
				->appends(request()->query());

			return EventResource::collection($query);
		}

		public function myRecommendations(Request $request){
			$user_group = $request->currentApiUser->group_id;

			$query = Event::selectRaw(self::EVENT_FIELDS . ', event_dates.date');



			$allowedFilters = [
				'events.name',
				'events.kind',
				'event_dates.date',
				'event_dates.date_end',
				'event_dates.location',
				'event_releases.released',
				'created_by',
				'recommendations.seen',
			];

			$allowedFilters = array_merge(Event::$filterable, $allowedFilters);

			foreach($request->filter as $value){
				if(in_array($value[0], $allowedFilters)){
					if($value[0] === 'event_dates.location' && $request->get('includeSubLocations', false)){
						$locations = Location::select('id')
							->where('parent_id', '=', $value[2])
							->get()
							->all();
						$locations = array_map(function($loc){
							return $loc->id;
						}, $locations);
						$locations = array_merge($locations, [$value[2]]);

						$query->whereIn($value[0], $locations);

						continue;
					}
					if($value[0] === 'event_releases.released'){
						$hasReleasedFilter = true;
						$query->where(function($query) use($value, $user_group){

							if($value[2] == '0'){
								$query->where(function($query) use ($user_group){
									$query->orWhere('event_releases.user_group', '=', $user_group);
									$query->orWhereNull('event_releases.user_group');
								});
								$query->where(function($query){
									$query->orWhereNull('event_releases.released');
									$query->orWhere('event_releases.released', '=', 0);
								});
							}else{
								$query->where('event_releases.user_group', '=', $user_group);
								$query->where('event_releases.released', '=', 1);
							}
						});
						continue;
					}

					if($value[0] === 'recommendations.seen'){
						$value[2] = (int) $value[2];
					}
					if($value[0] === 'event_dates.date_end'){
						$query->where(function($query) use ($value){
							$query->orWhere(...$value);
							$from = $value;
							$from[0] = 'event_dates.date';
							$query->orWhere(...$from);
						});
						continue;
					}
					$query->where(...$value);
				}
			}

			$query->whereNull('event_dates.deleted_at');

			$query->join('recommendations', 'events.id', '=', 'recommendations.event');

			$query->leftJoin('event_releases', function($join){
				$join->on('recommendations.event', '=', 'event_releases.event')
					->on('recommendations.user_group', '=', 'event_releases.user_group');
			});

			$query->where('recommendations.user_group', '=', $user_group);

			$query->leftJoin('event_dates', 'events.id', '=', 'event_dates.event');

			$query->groupByRaw(self::EVENT_FIELDS . ', event_dates.date');

			$query->orderBy('event_dates.date', 'asc');
			$query = $query->paginate($request->pageSize)
				->appends(request()->query());

			return EventResource::collection($query);
		}

		public function recommendationSeen(Request $request){
			$params = $request->all();
			$user_group = $request->currentApiUser->group_id;
			$recommendation = Recommendation::select()
				->where([
					['user_group', '=', $user_group],
					['event', '=', $params['event']],
				])
				->get()
				->first();

			if($recommendation){
				$recommendation->seen = 1;
				$recommendation->save();
			}

			return response()->json(null, 204);
		}

		public function recommendationUnseen(Request $request){
			$params = $request->all();
			$user_group = $request->currentApiUser->group_id;
			$recommendation = Recommendation::select()
				->where([
					['user_group', '=', $user_group],
					['event', '=', $params['event']],
				])
				->get()
				->first();

			if($recommendation){
				$recommendation->seen = 0;
				$recommendation->save();
			}

			return response()->json(null, 204);
		}

	}
	//loc 4274
	//date 1
